#include <glad/glad.h>
// glad.h must be included before glfw3.h
#include <GLFW/glfw3.h>

#include <iostream>
#include <string>
#include <filesystem>
#include <fstream>
#include <iterator>
#include <optional>
#include <vector>

namespace fs = std::filesystem;

std::optional<std::string> read_to_string(fs::path const &path) {
    if (std::ifstream ifs{path}; ifs) {
        return std::string(std::istreambuf_iterator<char>{ifs}, {});
    } else {
        return std::nullopt;
    }
}

void print_shader_compilation_errors(GLuint shader, fs::path const &path) {
    GLint log_length;
    glGetShaderiv(shader, GL_INFO_LOG_LENGTH, &log_length);
    if (log_length > 0) {
        std::vector<GLchar> log(log_length);
        glGetShaderInfoLog(shader, log.size(), &log_length, log.data());
        std::cerr << path.string() << ':' << log.data() << '\n';
    }
}

std::optional<GLuint> compile_shader(fs::path const &path, GLenum type) {
    if (auto shader_source = read_to_string(path); shader_source) {
        GLuint shader = glCreateShader(type);
        GLchar const *srcs[] = { shader_source->c_str() };
        glShaderSource(shader, 1, srcs, nullptr);
        glCompileShader(shader);
        
        GLint success;
        glGetShaderiv(shader, GL_COMPILE_STATUS, &success);
        if (success) {
            return shader;
        } else {
            print_shader_compilation_errors(shader, path);
            return std::nullopt;
        }
    } else {
        std::cerr << "failed read file: " << path << '\n';
        return std::nullopt;
    }
}

std::optional<GLuint> link_shaders(GLuint vert, GLuint frag) {
    GLuint program = glCreateProgram();
    glAttachShader(program, vert);
    glAttachShader(program, frag);
    glLinkProgram(program);
    
    GLint success;
    glGetProgramiv(program, GL_LINK_STATUS, &success);
    if (success) {
        return program;
    } else {
        return std::nullopt;
    }
}

std::optional<GLuint> shader_program_setup() {
    auto vert = compile_shader("shaders/hello_triangle.vert", GL_VERTEX_SHADER);
    auto frag = compile_shader("shaders/hello_triangle.frag", GL_FRAGMENT_SHADER);
    
    auto program = vert && frag ?
        link_shaders(*vert, *frag) : std::nullopt;
    
    if (vert) {
        glDeleteShader(*vert);
    }
    if (frag) {
        glDeleteShader(*frag);
    }
    
    return program;
}

int main() {
    glfwInit();
    glfwWindowHint(GLFW_CONTEXT_VERSION_MAJOR, 4);
    glfwWindowHint(GLFW_CONTEXT_VERSION_MINOR, 1);
    glfwWindowHint(GLFW_OPENGL_PROFILE, GLFW_OPENGL_CORE_PROFILE);
    glfwWindowHint(GLFW_OPENGL_FORWARD_COMPAT, GL_TRUE);
    
    GLFWwindow *window = glfwCreateWindow(800, 600, "Hello Triangle!", nullptr, nullptr);
    if (!window) {
        std::cerr << "GLFW failed to create window\n" << std::endl;
        return 1;
    }
    
    glfwMakeContextCurrent(window);
    
    if (!gladLoadGLLoader((GLADloadproc)glfwGetProcAddress)) {
        std::cerr << "glad failed to initialize OpenGL context\n";
        return 1;
    }
    
    auto *ver_gl = glGetString(GL_VERSION);
    std::cout << "OpenGL version: " << ver_gl << '\n';
    
    auto shader_program = shader_program_setup();
    if (!shader_program) {
        std::cerr << "failed to setup shader program\n";
        return 1;
    }
    
    GLuint VAO;
    glGenVertexArrays(1, &VAO);
    glBindVertexArray(VAO);
    
    while (!glfwWindowShouldClose(window)) {
        glClearColor(0.0, 0.0, 1.0, 1.0);
        glClear(GL_COLOR_BUFFER_BIT);
        
        glUseProgram(*shader_program);
        glDrawArrays(GL_TRIANGLES, 0, 3);
        
        glfwSwapBuffers(window);
        glfwPollEvents();
    }
    
    std::cerr << "Program finished successfully\n";
}
