#include <glad/glad.h>
// glad.h must be included before glfw3.h
#include <GLFW/glfw3.h>

#include "simpleshader.h"

#include <iostream>
#include <string>
#include <filesystem>
#include <fstream>
#include <iterator>
#include <optional>
#include <vector>

namespace fs = std::filesystem;
using namespace simpleshader;

std::optional<ShaderProgram> setup_shaders() {
    auto const vert_path = "shaders/hello_triangle.vert";
    auto const frag_path = "shaders/hello_triangle.frag";

    auto vert_src = read_to_string(vert_path);
    if (!vert_src) {
        std::cerr << "Error: failed to read file " << vert_path << '\n';
        return {};
    }

    auto frag_src = read_to_string(frag_path);
    if (!frag_src) {
        std::cerr << "Error: failed to read file " << frag_path << '\n';
        return {};
    }

    ShaderObject vert{GL_VERTEX_SHADER, *vert_src};
    if (!vert.ok()) {
        print_compilation_errors(vert, vert_path);
        return {};
    }

    ShaderObject frag{GL_FRAGMENT_SHADER, *frag_src};
    if (!frag.ok()) {
        print_compilation_errors(frag, frag_path);
        return {};
    }

    ShaderProgram program{vert, frag};
    if (!program.ok()) {
        print_link_errors(program);
        return {};
    }

    return program;
}

int main() {
    glfwInit();
    glfwWindowHint(GLFW_CONTEXT_VERSION_MAJOR, 4);
    glfwWindowHint(GLFW_CONTEXT_VERSION_MINOR, 1);
    glfwWindowHint(GLFW_OPENGL_PROFILE, GLFW_OPENGL_CORE_PROFILE);
    glfwWindowHint(GLFW_OPENGL_FORWARD_COMPAT, GL_TRUE);
    
    GLFWwindow *window = glfwCreateWindow(800, 800, "Hello Triangle!", nullptr, nullptr);
    if (!window) {
        std::cerr << "Error: GLFW failed to create window\n" << std::endl;
        glfwTerminate();
        std::exit(1);
    }
    
    glfwMakeContextCurrent(window);
    
    if (!gladLoadGLLoader((GLADloadproc)glfwGetProcAddress)) {
        std::cerr << "Error: glad failed to initialize OpenGL context\n";
        glfwTerminate();
        std::exit(1);
    }
    
    auto const version = glGetString(GL_VERSION);
    std::cout << "OpenGL version: " << version << '\n';
    
    auto shader_program = setup_shaders();
    if (!shader_program) {
        std::cerr << "Error: failed to setup shader program\n";
        glfwTerminate();
        std::exit(1);
    }
    
    GLuint VAO;
    glGenVertexArrays(1, &VAO);
    glBindVertexArray(VAO);
    
    while (!glfwWindowShouldClose(window)) {
        glClearColor(0.0, 0.0, 0.0, 1.0);
        glClear(GL_COLOR_BUFFER_BIT);
        
        glUseProgram(shader_program->name());
        glDrawArrays(GL_TRIANGLES, 0, 3);
        
        glfwSwapBuffers(window);
        glfwPollEvents();
    }

    shader_program->clear();

    glfwTerminate();

    std::clog << "Program finished successfully\n";
}
