#include "simpleshader.h"

#include <filesystem>
#include <fstream>
#include <iostream>
#include <optional>
#include <string>
#include <vector>

namespace fs = std::filesystem;

namespace /*anon*/ {

bool compile_shader(GLuint name, GLchar const *src) {
    GLchar const *src_arr[] = {src};
    glShaderSource(name, 1, src_arr, nullptr);
    glCompileShader(name);

    GLint success;
    glGetShaderiv(name, GL_COMPILE_STATUS, &success);
    return success;
}

bool link_shaders(GLuint program, GLuint vert, GLuint frag) {
    glAttachShader(program, vert);
    glAttachShader(program, frag);

    glLinkProgram(program);

    glDetachShader(program, frag);
    glDetachShader(program, vert);

    GLint success;
    glGetProgramiv(program, GL_LINK_STATUS, &success);
    return success;
}

}  // namespace

namespace simpleshader {

std::optional<std::string> read_to_string(fs::path const &path) {
    if (std::ifstream ifs{path}) {
        return std::string(std::istreambuf_iterator<char>{ifs}, {});
    } else {
        return std::nullopt;
    }
}

void print_compilation_errors(ShaderObject const &shader,
                              fs::path const &path) {
    GLint log_length;
    glGetShaderiv(shader.name(), GL_INFO_LOG_LENGTH, &log_length);
    if (log_length > 0) {
        std::vector<GLchar> log(log_length);
        glGetShaderInfoLog(shader.name(), log.size(), nullptr, log.data());
        std::cerr << path.string() << ':' << log.data() << '\n';
    }
}

void print_link_errors(ShaderProgram const &program) {
    GLint log_length;
    glGetProgramiv(program.name(), GL_INFO_LOG_LENGTH, &log_length);
    if (log_length > 0) {
        std::vector<GLchar> log(log_length);
        glGetProgramInfoLog(program.name(), log.size(), nullptr, log.data());
        std::cerr << log.data() << '\n';
    }
}

ShaderObject::ShaderObject(GLenum type, std::string const &src)
        : type_{type}, name_{} {
    name_ = glCreateShader(type_);
    compile_shader(name_, src.c_str());
}

ShaderObject::~ShaderObject() {
    glDeleteShader(name_);
}

ShaderObject::ShaderObject(ShaderObject &&other) noexcept
        : type_{other.type_}, name_{other.name_} {
    type_ = 0;  // nothing?
    name_ = 0;
}

ShaderObject &ShaderObject::operator=(ShaderObject &&other) noexcept {
    using std::swap;
    swap(type_, other.type_);
    swap(name_, other.name_);
    return *this;
}

bool ShaderObject::ok() const {
    GLint success;
    glGetShaderiv(name_, GL_COMPILE_STATUS, &success);
    return success;
}

GLuint ShaderObject::name() const {
    return name_;
}

ShaderProgram::ShaderProgram(ShaderObject const &vert, ShaderObject const &frag)
        : name_{} {
    name_ = glCreateProgram();
    link_shaders(name_, vert.name(), frag.name());
}

ShaderProgram::~ShaderProgram() {
    glDeleteProgram(name_);
}

ShaderProgram::ShaderProgram(ShaderProgram &&other) noexcept
        : name_{other.name_} {
    other.name_ = 0;
}

ShaderProgram &ShaderProgram::operator=(ShaderProgram &&other) noexcept {
    using std::swap;
    swap(name_, other.name_);
    return *this;
}

bool ShaderProgram::ok() const {
    GLint success;
    glGetProgramiv(name_, GL_LINK_STATUS, &success);
    return success;
}

GLuint ShaderProgram::name() const {
    return name_;
}

void ShaderProgram::clear() {
    glDeleteProgram(name_);
    name_ = 0;
}

}  // namespace simpleshader
